;+
; Name        :	FXINHERIT()
;
; Purpose     :	Inherit keywords from primary header.
;
; Category    :	FITS
;
; Explanation :	This routine forms a new header from the combination of
;               keywords from the primary header and extension header.
;
; Syntax      :	New_Header = FXINHERIT(Primary_Header, Ext_Header)
;
; Examples    :	OPENR, UNIT, FILE, /GET_LUN
;               FXHREAD, UNIT, PRIMARY_HEADER
;               FREE_LUN, UNIT
;               FXREAD, FILE, IMAGE, EXT_HEADER, EXTENSION=1
;               HEADER = FXINHERIT(PRIMARY_HEADER, EXT_HEADER)
;
; Inputs      :	PRIMARY_HEADER = Primary header of FITS file
;               EXT_HEADER     = Extension header
;
; Outputs     :	The result of the function is the combined header.
;
; History     :	Version 1, William Thompson, 10 March 2025
;
; Contact     :	WTHOMPSON
;-
;
function fxinherit, primary_header, ext_header
;
;  Get the keywords from the two headers.
;
key0 = strupcase(strmid(primary_header, 0, 8))
key1 = strupcase(strmid(ext_header, 0, 8))
;
;  Form a new header from the extension header, up to the END keyword.
;
w1 = (where(key1 eq 'END     '))[0]
header = ext_header[0:w1-1]
;
;  Add in the keywords from the primary header, so long as they are not already
;  in the extension header.  Skip the SIMPLE keyword at the beginning, and any
;  COMMENT, HISTORY or blank keywords.
;
for i = 1,n_elements(key0)-2 do begin
    k0 = key0[i]
    if (k0 ne 'COMMENT ') and (k0 ne 'HISTORY ') and (k0 ne '        ') $
    then begin
        w = where(k0 eq key1, count)
        if count eq 0 then header = [header, primary_header[i]]
    endif
endfor
;
;  End the END keyword, and return the header.    
;
header = [header, ext_header[w1]]
return, header
end
