;+
; Name: framework_insert_catch
;
; Project: HESSI
;
; Purpose: Insert error handlers in HESSI objects.
;
; Method:
; This code should be inserted (via @) into any FRAMEWORK GETDATA or GET method in which
; we want to establish an error handler. It should be placed at the beginning of the
; method.  @ must be used (rather than making this a procedure) so that the catch will
; actually be IN the routine we want to stop in.  The added line should look like:
;   @framework_insert_catch
;
; Any exception that occurs in any routine called by GETDATA or GET will jump back
; to the nearest error handler (looking up the traceback backward).  Exceptions are
; caused in two ways and are handled differently in the catch:
;
;   If the exception was caused by a MESSAGE statement, the message is printed.
;   If the exception was caused by an error in the code, the error message and
;   traceback are printed.
;
; If the error handler is within a series of calls to framework get or getdata, then
; it is assumed that there is another error handler further up the chain, and this
; error handler generates another exception via the MESSAGE routine which will bubble
; up to the next error handler.
; If the error handler is not nested inside FRAMEWORK object calls, then this error handler
; prints a message that it's aborting, and returns a -1.
;
; Since this error handler returns a -1, it can only be inserted in a function. (We
; can make another version if we need to insert it into a procedure.)
;
; In routines that encounter error conditions that should cause an abort of the current
; operation, there should be a call to HSI_MESSAGE (or MESSAGE without a /CONTINUE or /INFO)
;
; For programmers who want to debug without the catch, call framework_debug,10 to set the debug level to 10:
; If ssw_framework_debug() is not zero, then catch is NOT set to trap errors.
; In that case, errors will cause normal failures (instead of jumping to the nearest catch)
; and programmers can debug normally.
;
; See HSI_MESSAGE also.
;
; Written: Kim Tolbert, 20-Aug-2002
; Modifications:
;  2007-aug-07, Kim changed hsi_insert_catch to framework_insert_catch, and call framework_is_obj_nested
;  2013-May-31, Kim. Instead of checking self.debug, check framework_get_debug() (which looks at env. var.
;    SSW_FRAMEWORK_DEBUG) to decide whether to set catch. self.debug only gets set when the object is created,
;    whereas SSW_FRAMEWORK_DEBUG can be set anytime to make code stop on an error (for debugging) or follow 
;    the catch (which handles the error, and does something 'smart').
;-
;-----------------------------------------------------------------------------------------

error_catch = 0
;if self.debug eq 0 then catch, error_catch
if framework_get_debug() eq 0 then catch, error_catch
if error_catch ne 0 then begin

	catch, /cancel

	; was exception generated by a MESSAGE call?
	if !error_state.name eq 'IDL_M_USER_ERR' then print,!error_state.msg else begin $
		help, /last_message, out=out
		prstr,out, /nomore
	endelse

	; calldetails.module will contain the name of the current procedure
	;calldetails = get_calldetails(1)
	msg = 'Aborting ' ; + calldetails.module

	; if we're nested then generate an exception via MESSAGE to jump to previous handler
	if framework_is_obj_nested() then message, msg

	; otherwise, use MESSAGE to show that we're aborting, but then return a -1
	message, msg + '.  Returning -1.', /cont
	return, -1

endif