pro make_mirror, site, remote_dir, local_dir,                             $
                packages=packages, comments=comments,                     $
                remote_user=remote_user, remote_password=remote_password, $ 
                mirror_file=mirror_file,mode_copy=mode_copy,              $
                passive_ftp=passive_ftp, $
                use_timelocal=use_timelocal, _extra=_extra,               $
                user=user, group=group, $
                nouser=nouser, nogroup=nogroup, $
                nosymbolic_links=nosymbolic_links, $
                max_delete_files=max_delete_files, $
                max_delete_dirs=max_delete_dirs,   $
                no_symbolic_links=no_symbolic_links, $
                update_log=update_log, noduplicate_last=noduplicate_last
;+
;   Name: make_mirror
;
;   Purpose: build a mirror file
;
;   Input Parameters:
;      site       - remote site / ip
;      remote_dir - list of remote_dirs to mirror
;      local_dir  - list of local dirs  (destination) 
;
;   Keyword Parameters:
;      remote_user -   (default = anonymous)
;      remote_password (default = user@local_host) 
;      packages - if passed, strarray containing package names (one per remote dir)
;      comments - is passed, strarray containing comments (one per remote dir)
;      mirror_file - output mirror file name w/path (default= "mk_mirror" in current)
;      no_symbolic_links - if set, dont follow remote links 
;      XXX=NN      - anyother where XXX=anyother mirror keword
;                    and             NN=associated value
;   
;   Calling Sequence:
;      make_mirror, siteip, remote_dir, local_dir [,mirror_file='filename', $
;                           packages=[package_names]
;
;   Calling Example:
;      IDL> <run routine to define remote and local directory lists>      
;      IDL> make_mirror,'sohoftp.nascom.nasa.gov',remotedirs, localdirs, mirror_file='SSW.mirror'
;      IDL> make_mirror,ip,remote,local,MAX_DAYS=10  ; recent 10 days only
;                                                    ; (via keyword inherit)  
;
;   18-march-1996 (S.L.Freeland) - originally for remote SSW install/upgrades
;   21-Oct-96   RDB     Added comments keyword
;   21-Oct-96   S.L.Freeland - change compress_pat to compress_patt
;   23-Oct-96   S.L.Freeland - add /MODE_COPY keyword
;   12-nov-96   S.L.Freeland - add USE_TIMELOCAL (default=false)
;   31-mar-97   S.L.Freeland - add USER and GROUP keywords
;   22-may-97   S.L.Freeland - add MAX_DELETE_FILES  and MAX_DELETE_DIRS
;    4-feb-98   S.L.Freeland - init MAX_DELETE_FILES MAX_DELETE_DIRS if 
;                              undefined (set to 10%, the mirror default)
;   20-mar-98   S.L.Freeland - enabled the keyword inheritance part
;    2-Feb-98   S.L.Freeland - include symbolic link following by default
;   22-Feb-98   S.L.Freeland - add /NO_SYMBOLIC_LINKS (override 2-feb default)
;    6-Dec-98   S.L.Freeland - add UPDATE_LOG keyword and function
;    8-Dec-98   S.L.Freeland - add /NOUSER and /NOGRROUP key/func
;                              made those the Windows default (local_dir)
;  
;   20-Mar-2000 S.L.Freeland - some windows stuff
;   30-Apr-2000 S.L.Freeland - due to Windows Mirror glitch, duplicate
;                              the last package
;    4-Oct-2001 S.L.Freeland - add /PASSIVE_FTP keyword and function
;   17-mar-2003 S.L.Freeland - added an EXCLUDE_PATT for Windows
;                              to avoid illegal DOS/Win filename from unix
;  
;   Restrictions:
;      must supply siteip and remote & local dirs (minimum)
;      assume one-one mapping between remote and local
;-
if not data_chk(site,/string) then begin
   message,/info,"Need to specify remote site..."
   return
endif

npacks=n_elements(remote_dir)

case 1 of 
   data_chk(remote_dir,/string) and data_chk(local_dir,/string): begin
      if npacks ne n_elements(local_dir) then begin
         message,/info,"number of remote and local dirs must match..."
         return
      endif 
   endcase
   else: begin
      message,/info,"Need to specify remote and local directory lists..."
      return
   endcase
endcase

; set some default field values
compress_patt=''
if n_elements(max_delete_files) eq 0 then max_delete_files='10%'
if n_elements(max_delete_dirs) eq 0 then max_delete_dirs='10%'

mcomment='Generated by make_mirror at: ' + systime()
comment=''
if not keyword_set(comments) then comment = mcomment
package= str_replace(remote_dir,'/','_')        
if not keyword_set(remote_user) then begin
   remote_user='anonymous'
   remote_password=get_user()+'@'+get_host()
endif

mode_copy=(['false','true'])(keyword_set(mode_copy)) ; use mirror default=false
passive_ftp=(['false','true'])(keyword_set(passive_ftp)) ; mirror def=false
use_timelocal=(['false','true'])(keyword_set(use_timelocal)) ; use mirror default=false

if not keyword_set(user) then user=get_user()
if not keyword_set(group) then group=''

; define minimal mirror fields
; ------------------ offer template file option ?? --------
packstr={package:'',                            $
         comment:'',                            $
         site:site,                             $
         remote_user:'',                        $
         remote_password:'',                    $
         remote_dir:'',                         $
         local_dir:'',                          $
         mode_copy:'',                          $
         passive_ftp:'',                        $
         use_timelocal:'',                      $
         user:'',                               $
         group:'',                              $
         max_delete_files:'',                   $
         max_delete_dirs:'',                    $
         update_log:'',                         $
         compress_patt:''}          
; ------------------ offer template file option ?? --------

windows=os_family() eq 'Windows' or $
                  strpos(local_dir(0),'\') ne -1
nogroup=keyword_set(nogroup) or windows
nouser=keyword_set(nouser) or windows
etags=['',(['','group'])(nogroup),(['','user'])(nouser),$
          (['','update_log'])(1-data_chk(update_log,/string))] ; except tags

sse=where(etags ne '',etcnt)
if etcnt gt 0 then begin
   box_message,'Eliminating tags: ' + arr2str(etags(sse))
   packstr=str_subset(packstr,etags(sse),/exclude)        ; eliminate these tags
endif
        
allpacks=replicate(packstr,npacks)              ; one per package
tags=strlowcase(tag_names(packstr))             ; info
ntags=n_tags(packstr)                           ; info

for i=0,n_tags(packstr)-1 do begin              ; assign tag values
   exe=execute('val=' + tags(i))
   exe=execute('allpacks.(i)=tags(i)+"="+val')
endfor

if n_elements(packages) eq npacks then allpacks.package='package='+packages
if n_elements(comments) eq npacks then allpacks.comment='comment='+comments
if n_elements(update_log) eq npacks then allpacks.update_log='update_log='+update_log

if not keyword_set(mirror_file) then mirror_file='mk_mirror'

if data_chk(_extra,/struct) then begin
   tnames=strlowcase(tag_names(_extra))
   for i=0,n_tags(_extra)-1 do allpacks=$
        add_tag(allpacks,tnames(i)+'='+strtrim(_extra.(i),2),tnames(i))
endif  
;
; 
if not keyword_set(no_symbolic_links) and (tag_index(allpacks,'flags_recursive'))(0) eq -1 then begin
   allpacks=add_tag(allpacks,'flags_recursive+L','flags_recursive')
endif

if windows then begin
   box_message,'Adding Windows tweaks...'

   arr=allpacks.remote_dir
   allpacks.remote_dir= $
       strmids(arr, 0,  strlen(arr)-(strlastchar(arr) eq '/'))+'/'
   
   arr=allpacks.local_dir
   allpacks.local_dir= $
       strmids(arr, 0,  strlen(arr)-(strlastchar(arr) eq '\'))


;  allpacks=add_tag(allpacks,'get_size_change=false','get_size_change')
   allpacks=add_tag(allpacks,'force_times=yes','force_times')

   allpacks=add_tag(allpacks,'exclude_patt=[:\?]','exclude_patt')

   if not keyword_set(noduplicate_last) then $      ; Windows/Mirror problem
         allpacks=[allpacks,last_nelem(allpacks)]   ; dupliate last package
endif 

ntags=n_tags(allpacks(0))
; write output mirror file 
form='(' + strtrim(npacks,2) + '(1h#,/,a,/,'+ strtrim(ntags-1,2) + '(3x,a,/)))'
file_append,mirror_file,'# '+mcomment,/new
file_append,mirror_file,string(allpacks,format=form)    ;, /new
message,/info,"Wrote mirror file: " + mirror_file

return
end
