; docformat = 'rst'

;+
; Linear interpolation in many dimensions.
;
; Like the bilinear and trilinear interpolation of IDL's built-in
; INTERPOLATE function. Wrapper for ninterpolate by J.D. Smith to
; support interpolation for multiple points.
; 
; :Categories:
;
;    SST pipeline
; 
; 
; :Author:
; 
;    Mats Löfdahl, Institute for Solar Physics
; 
; 
; :Returns:
; 
;    An array with the interpolated values.
;
; :Params:
; 
;    P : in, type=fltarr/dblarr
; 
;      The array of data values. The number of dimensions should match
;      the number of coordinate arrays.
;
;    x0 : in, type=fltarr/dblarr
;
;       Array with coordinate values for the first coordinate.
; 
;    x1 : in, type=fltarr/dblarr
;
;       Array with coordinate values for the second coordinate.
; 
;    x2 : in, type=fltarr/dblarr
;
;       Array with coordinate values for the third coordinate.
;    :
;    :
;
;    x7 : in, type=fltarr/dblarr
; 
; :Keywords:
; 
;    mode : in, optional, type=string
;   
;       If 'idl', use IDL's built-in interpolate function. If
;       'mint', use minterpolate.
; 
; 
; :History:
; 
;    2017-09-29 : MGL. First version.
; 
;    2024-08-20 : MGL. Double precision.
; 
;-
function interpolate_many, P, x0, x1, x2, x3, x4, x5, x6, mode = mode

  Ndims = size(P, /n_dim)
  Npoints = n_elements(x0) 

  if Ndims eq 0 or Npoints eq 0 then stop

  ;; No need for multiple dimensions 
  if Ndims eq 1 then return, interpolate(/double, p, x0)

  if n_elements(mode) eq 0 then mode = ''
  
  ;; Check same number of points in all axes
  if Ndims eq 2 and n_elements(x1) ne Npoints then stop
  if Ndims eq 3 and n_elements(x2) ne Npoints then stop
  if Ndims eq 4 and n_elements(x3) ne Npoints then stop
  if Ndims eq 5 and n_elements(x4) ne Npoints then stop
  if Ndims eq 6 and n_elements(x5) ne Npoints then stop
  if Ndims eq 7 and n_elements(x6) ne Npoints then stop

  ;; Want the built-in IDL interpolate?
  if mode eq 'idl' then begin
     case Ndims of
       2 : return, interpolate(/double, p, x0, x1)
       3 : return, interpolate(/double, p, x0, x1, x2)
       else : stop
     endcase
  endif

  case Ndims of
     2 : points = [[x0],[x1]]
     3 : points = [[x0],[x1],[x2]]
     4 : points = [[x0],[x1],[x2],[x3]]
     5 : points = [[x0],[x1],[x2],[x3],[x4]]
     6 : points = [[x0],[x1],[x2],[x3],[x4],[x5]]
     7 : points = [[x0],[x1],[x2],[x3],[x4],[x5],[x6]]
     else : stop
  endcase

  result = dblarr(Npoints)
  if mode eq 'mint' then $
     for ipoint = 0, Npoints-1 do result[ipoint] = minterpolate(P, reform(points[ipoint, *])) $
  else $
     for ipoint = 0, Npoints-1 do result[ipoint] = ninterpolate_ssw(P, reform(points[ipoint, *])) 
     

  return, result
  
end

data2 = randomu(/double, seed, 10, 10)
data3 = randomu(/double, seed, 10, 10, 10)
data4 = randomu(/double, seed, 10, 10, 10, 10)
data5 = randomu(/double, seed, 10, 10, 10, 10, 10)
data6 = randomu(/double, seed, 10, 10, 10, 10, 10, 10)

x0 = [.37d, .45d, .12d]*10
x1 = [.81d, .45d, .22d]*10
x2 = [.13d, .45d, .32d]*10
x3 = [.22d, .45d, .42d]*10
x4 = [.47d, .45d, .52d]*10
x5 = [.73d, .45d, .62d]*10

print
print, '2D'

n = interpolate_many(data2, x0, x1)
m = interpolate_many(data2, x0, x1, mode = 'mint')
i = interpolate_many(data2, x0, x1, mode = 'idl')

print, 'ninterpolate', n, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'minterpolate', m, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'iinterpolate', i, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, '         n-i', n-i
print, '         m-i', m-i
print, '         m-n', m-n
print

print
print, '3D'
n = interpolate_many(data3, x0, x1, x2)
m = interpolate_many(data3, x0, x1, x2, mode = 'mint')
i = interpolate_many(data3, x0, x1, x2, mode = 'idl')

print, 'ninterpolate', n, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'minterpolate', m, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'iinterpolate', i, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, '         n-i', n-i
print, '         m-i', m-i
print, '         m-n', m-n
print

print
print, '4D'
n = interpolate_many(data4, x0, x1, x2, x3)
m = interpolate_many(data4, x0, x1, x2, x3, mode = 'mint')

print, 'ninterpolate', n, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'minterpolate', m, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, '         m-n', m-n
print

print
print, '4D'
n = interpolate_many(data5, x0, x1, x2, x3, x4)
m = interpolate_many(data5, x0, x1, x2, x3, x4, mode = 'mint')
print, 'ninterpolate', n, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'minterpolate', m, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, '         m-n', m-n
print

print
print, '6D'
n = interpolate_many(data6, x0, x1, x2, x3, x4, x5)
m = interpolate_many(data6, x0, x1, x2, x3, x4, x5, mode = 'mint')
print, 'ninterpolate', n, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, 'minterpolate', m, format = '(a0, f21.17, f21.17, f21.17, f21.17)'
print, '         m-n', m-n
print

 
                
end            
