; docformat = 'rst'

;+
; Perform n-dimensional linear interpolation on arrays of arbitrary
; dimensionality (a la the bilinear and trilinear interpolation of
; INTERPOLATE). Used to verify the less readable ninterpolate_ssw.
; 
; :Categories:
;
;    Maths
; 
; 
; :Author:
; 
;    Mats Löfdahl, Institute for Solar Physics
; 
; 
; :Returns:
;
;    The interpolated value.
; 
; :Params:
; 
;    data : in, type=fltarr(N1,N2, ..., Nn)
;
;       The array to interpolate.
; 
;    point : in, type=fltarr(n)
;
;       A vector specifying the (single) point for which the
;       interpolated value is desired.
; 
; :History:
; 
;    2017-10-02 : MGL. First version.
; 
;    2024-08-20 : MGL. Double precision.
;
;-
function minterpolate, data, point

  n = n_elements(point)

  if n eq 0 then stop

  wsum = 0d
  vol = 0d
  
  for i0 = 0, 1 do begin
    
    if i0 then c0 = [floor(point[0]), ceil(point[0])] else c0 = [ceil(point[0]), floor(point[0])]

    if n eq 1 then begin
      
      opoint = [c0[1]]                      ; Opposite corner
      subvol = abs(product(opoint-point))   ; Volume of subcube between point and opposite corner
      vol += subvol
      wsum += data[c0[0]] * subvol

    endif else for i1 = 0, 1 do begin
      
      if i1 then c1 = [floor(point[1]), ceil(point[1])] else c1 = [ceil(point[1]), floor(point[1])]
      
      if n eq 2 then begin

        opoint = [c0[1], c1[1]]
        subvol = abs(product(opoint-point))
        vol += subvol
        wsum += data[c0[0], c1[0]] * subvol

      endif else for i2 = 0, 1 do begin
        
        if i2 then c2 = [floor(point[2]), ceil(point[2])] else c2 = [ceil(point[2]), floor(point[2])]
        
        if n eq 3 then begin

          opoint = [c0[1], c1[1], c2[1]]
          subvol = abs(product(opoint-point))
          vol += subvol
          wsum += data[c0[0], c1[0], c2[0]] * subvol

        endif else for i3 = 0, 1 do begin
          
          if i3 then c3 = [floor(point[3]), ceil(point[3])] else c3 = [ceil(point[3]), floor(point[3])]
          
          if n eq 4 then begin

            opoint = [c0[1], c1[1], c2[1], c3[1]]
            subvol = abs(product(opoint-point))
            vol += subvol
            wsum += data[c0[0], c1[0], c2[0], c3[0]] * subvol

          endif else for i4 = 0, 1 do begin
            
            if i4 then c4 = [floor(point[4]), ceil(point[4])] else c4 = [ceil(point[4]), floor(point[4])]
            
            if n eq 5 then begin

              opoint = [c0[1], c1[1], c2[1], c3[1], c4[1]]
              subvol = abs(product(opoint-point))
              vol += subvol
              wsum += data[c0[0], c1[0], c2[0], c3[0], c4[0]] * subvol

            endif else for i5 = 0, 1 do begin
              
              if i5 then c5 = [floor(point[5]), ceil(point[5])] else c5 = [ceil(point[5]), floor(point[5])]
              
              if n eq 6 then begin

                opoint = [c0[1], c1[1], c2[1], c3[1], c4[1], c5[1]]
                subvol = abs(product(opoint-point))
                vol += subvol
                wsum += data[c0[0], c1[0], c2[0], c3[0], c4[0], c5[0]] * subvol

              endif else for i6 = 0, 1 do begin
                
                if i6 then c6 = [floor(point[6]), ceil(point[6])] else c6 = [ceil(point[6]), floor(point[6])]
                
                if n eq 7 then begin

                  opoint = [c0[1], c1[1], c2[1], c3[1], c4[1], c5[1], c6[1]]
                  subvol = abs(product(opoint-point))
                  vol += subvol
                  wsum += data[c0[0], c1[0], c2[0], c3[0], c4[0], c5[0], c6[0]] * subvol

                endif else for i7 = 0, 1 do begin
                  
                  if i7 then c7 = [floor(point[7]), ceil(point[7])] else c7 = [ceil(point[7]), floor(point[7])]

                  if n eq 8 then begin
                    opoint = [c0[1], c1[1], c2[1], c3[1], c4[1], c5[1], c6[1], c7[1]]
                    subvol = abs(product(opoint-point))
                    vol += subvol
                    wsum += data[c0[0], c1[0], c2[0], c3[0], c4[0], c5[0], c6[0], c7[0]] * subvol
                  endif else begin
                    print, 'Too many dimensions.'
                    stop
                  endelse
                  
                endfor          ; i7
              endfor            ; i6
            endfor              ; i5
          endfor                ; i4
        endfor                  ; i3
      endfor                    ; i2
    endfor                      ; i1
  endfor                        ; i0

  return, wsum

end


;; Testing

for n = 1, 8 do begin

  print
  print, '# of dimensions: ', n
  
  dims = replicate(2, n)
  r = randomu(sd,dims)
  point = randomu(sd,n)*(dims-1)

  if n le 3 then begin
    clock=tic() 
    case n of
      1 : iint = interpolate(/double, r, point, /double)
      2 : iint = interpolate(/double, r, point[0], point[1], /double)
      3 : iint = interpolate(/double, r, point[0], point[1], point[2], /double)
    endcase
    time = toc(clock)
    print, ' interpolate: ', string(iint, format = '(f11.9)'), ' in '+string(time*1e6, format = '(f8.3)')+ ' µs'
  endif
  clock=tic() 
  mint = minterpolate(r, point)
  time = toc(clock)
  print, 'Minterpolate: ', string(mint, format = '(f11.9)'), ' in '+string(time*1e6, format = '(f8.3)')+ ' µs'
  clock=tic() 
  nint = ninterpolate(r,point)
  time = toc(clock)
  print, 'Ninterpolate: ', string(nint, format = '(f11.9)'), ' in '+string(time*1e6, format = '(f8.3)')+ ' µs'

  
  
endfor                          ; n

end

