;+
; Project     :	STEREO
;
; Name        :	WCS_SPEC2IMAGE()
;
; Purpose     :	Removes non-spatial dimensions from a WCS structure.
;
; Category    :	FITS, Coordinates, WCS
;
; Explanation :	This procedure modifies a WCS structure to remove non-spatial
;               dimensions such as wavelength and time.  The supposition is
;               that the original WCS structure applies to spectral data which
;               has been analyzed to produce a two-dimensional image such as
;               intensity, velocity, linewidth, etc.  A new WCS structure is
;               returned in which information about the non-spatial dimensions
;               is removed.
;
; Syntax      :	Result = WCS_SPEC2IMAGE(WCS)
;
; Inputs      :	WCS  = World Coordinate System structure from FITSHEAD2WCS
;
; Opt. Inputs :	None.
;
; Outputs     :	The result of the function is a modified WCS structure.
;
; Opt. Outputs:	None.
;
; Keywords    :	None.
;
; Calls       :	VALID_WCS, TAG_EXIST, REP_TAG_VALUE, WCS_DECOMP_ANGLE, REM_TAG,
;               WCS_SIMPLE
;
; Common      :	None.
;
; Restrictions:	Assumes that only the non-spatial dimensions are collapsed, and
;               the data retains the original spatial dimensions.
;
; Side effects:	Any information about distortions are lost.
;
; Prev. Hist. :	None.
;
; History     :	Version 1, 10-Feb-2025, William Thompson, GSFC
;
; Contact     :	WTHOMPSON
;-
;
function wcs_spec2image, wcs, _extra=_extra
on_error, 2
;
;
;  Check the input parameters.
;
if n_params() ne 1 then message, 'Syntax: Result = WCS_SPEC2IMAGE(WCS)'
if not valid_wcs(wcs) then message, 'Input not recognized as WCS structure'
;
;  Determine which axes are spatial.
;
naxis = n_elements(wcs.naxis)
spatial = bytarr(naxis)
spatial[wcs.ix] = 1
spatial[wcs.iy] = 1
wspatial = where(spatial eq 1)
;
;  Modify the information in the NEW_WCS structure.
;
new_wcs = wcs
if tag_exist(new_wcs, 'NAXIS') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.naxis[wspatial], 'NAXIS')
if tag_exist(new_wcs, 'CRPIX') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.crpix[wspatial], 'CRPIX')
if tag_exist(new_wcs, 'CRVAL') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.crval[wspatial], 'CRVAL')
if tag_exist(new_wcs, 'CTYPE') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.ctype[wspatial], 'CTYPE')
if tag_exist(new_wcs, 'CNAME') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.cname[wspatial], 'CNAME')
if tag_exist(new_wcs, 'CUNIT') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.cunit[wspatial], 'CUNIT')
if tag_exist(new_wcs, 'CDELT') then $
  new_wcs = rep_tag_value(new_wcs, new_wcs.cdelt[wspatial], 'CDELT')
;
;  If there is a PC or CD matrix, then keep only the spatial components.
;
if tag_exist(new_wcs, 'PC') then begin
    pc = new_wcs.pc[wspatial,*]
    new_wcs = rep_tag_value(new_wcs, pc[*,wspatial], 'PC')
    wcs_decomp_angle, new_wcs, roll_angle, cdelt, found, /add_tags
endif
if tag_exist(new_wcs, 'CD') then begin
    cd = new_wcs.cd[wspatial,*]
    new_wcs = rep_tag_value(new_wcs, cd[*,wspatial], 'CD')
    wcs_decomp_angle, new_wcs, roll_angle, cdelt, found, /add_tags
endif
;
;  Remove any SPECTRUM or DISTORTION tags.
;
if tag_exist(new_wcs, 'SPECTRUM') then $
  new_wcs = rem_tag(new_wcs, 'SPECTRUM')
if tag_exist(new_wcs, 'DISTORTION') then $
  new_wcs = rem_tag(new_wcs, 'DISTORTION')
;
;  Determine whether or not the new WCS structure is simple, and return it.
;
new_wcs.simple = wcs_simple(new_wcs)
return, new_wcs
end
